//@author: dottore
//@help: this shader apply tone mappping to the hdr source map using luminance texture
//@tags: tone mapping hdr
//@credits:

// --------------------------------------------------------------------------------------------------
// PARAMETERS:
// --------------------------------------------------------------------------------------------------

//transforms
float4x4 tW: WORLD;        //the models world matrix
float4x4 tV: VIEW;         //view matrix as set via Renderer (EX9)
float4x4 tP: PROJECTION;
float4x4 tWVP: WORLDVIEWPROJECTION;

//texture
texture TexOrig <string uiname="Original Texture";>;
sampler TexToToneMap = sampler_state    //sampler for doing the texture-lookup
{
    Texture   = (TexOrig);          //apply a texture to the sampler
    MipFilter = none;         //sampler states
    MinFilter = none;
    MagFilter = none;
};

texture TexLw <string uiname="Luma Texture";>;
sampler BarLw = sampler_state    //sampler for doing the texture-lookup
{
    Texture   = (TexLw);          //apply a texture to the sampler
    MipFilter = none;         //sampler states
    MinFilter = none;
    MagFilter = none;
};

uniform  float RecipLwhiteSqrd;

//the data structure: "vertexshader to pixelshader"
//used as output data with the VS function
//and as input data with the PS function
struct vs2ps
{
    float4 Pos  : POSITION ;
    float2 TexCd : TEXCOORD0 ;
    float Luma: TEXCOORD1 ;
};

// --------------------------------------------------------------------------------------------------
// VERTEXSHADERS
// --------------------------------------------------------------------------------------------------
vs2ps VS(
    float4 PosO  : POSITION,
    float4 TexCd : TEXCOORD0)
{
    //declare output struct
    vs2ps Out;

    //transform position
    Out.Pos = mul(PosO, tWVP);
    
    //transform texturecoordinates
    Out.TexCd = TexCd;
    
    Out.Luma = tex2Dlod( BarLw, 0 );

    return Out;
}

// --------------------------------------------------------------------------------------------------
// PIXELSHADERS:
// --------------------------------------------------------------------------------------------------

float4 PS(vs2ps In):COLOR
          //uniform in float RecipA,
          //uniform in float RecipLwhiteSqrd)
{
    // get avarage Lw from the texture we calculate previously
    float aOverBarLw = In.Luma;
    
    // calc L(x,y) by getting the Luminance of this pixel and
    // scaling by a/BarLw
    float4 hdrPixel = tex2D(TexToToneMap, In.TexCd);
    float lumi = dot( hdrPixel, float3( 0.27, 0.67, 0.06 ) );
    float Lxy = aOverBarLw * lumi;
    
    // now calculate Ld
    float numer = (1 + (Lxy * RecipLwhiteSqrd)) * Lxy;
    float denum = 1 + Lxy;
    float Ld = numer / denum;
    
    // we now have to convert the original color into this range
    float3 ldrPixel = (hdrPixel / lumi) * Ld;
    
    // currently don't process hdr alpha just saturate on the output to LDR
    return float4(ldrPixel, hdrPixel.a);

}

// --------------------------------------------------------------------------------------------------
// TECHNIQUES:
// --------------------------------------------------------------------------------------------------

technique TSimpleShader
{
    pass P0
    {
        //Wrap0 = U;  // useful when mesh is round like a sphere
        VertexShader = compile vs_3_0 VS();
        PixelShader  = compile ps_2_0 PS();
    }
}
